import { GetObjectCommand } from "@aws-sdk/client-s3";
import { getSignedUrl } from "@aws-sdk/s3-request-presigner";
import { s3Client, R2_BUCKET_NAME } from "../config/r2.config";
import { idriveClient, IDRIVE_BUCKET_NAME } from "../config/idrive.config";
import { Readable } from "stream";
import https from "https";
import http from "http";
import axios from "axios";

/**
 * Supported storage providers for proxy streaming.
 */
export type DownloadProvider = "r2" | "pixeldrain" | "idrive" | "vikingfile";

/**
 * Represents the downloadable context for a single provider.
 */
export interface ProviderDownloadContext {
  provider: DownloadProvider;
  /** The stored identifier (S3 key, file hash, etc.) */
  identifier: string;
  /** The original stored URL (used as fallback for VikingFile) */
  storedUrl?: string;
}

/**
 * Result from a proxy stream resolution — a readable stream plus content metadata.
 */
export interface ProxyStreamResult {
  stream: Readable;
  contentType: string;
  contentLength?: number;
  filename: string;
}

/**
 * Service responsible for fetching/streaming files from each provider
 * through the backend, bypassing hotlinks, private-bucket restrictions, etc.
 */
export class DownloadService {
  private static readonly PRESIGNED_URL_TTL_SECONDS = 300; // 5 minutes

  /**
   * Generates a pre-signed S3 GET URL for Cloudflare R2.
   */
  static async getR2PresignedUrl(r2Key: string): Promise<string> {
    const command = new GetObjectCommand({
      Bucket: R2_BUCKET_NAME,
      Key: r2Key,
    });
    return getSignedUrl(s3Client, command, {
      expiresIn: this.PRESIGNED_URL_TTL_SECONDS,
    });
  }

  /**
   * Generates a pre-signed S3 GET URL for IDrive e2.
   */
  static async getIdrivePresignedUrl(idriveKey: string): Promise<string> {
    const command = new GetObjectCommand({
      Bucket: IDRIVE_BUCKET_NAME,
      Key: idriveKey,
    });
    return getSignedUrl(idriveClient, command, {
      expiresIn: this.PRESIGNED_URL_TTL_SECONDS,
    });
  }

  /**
   * Resolves and streams a file from the given provider.
   * Returns a raw Node.js Readable plus metadata for the proxy handler.
   */
  static async resolveStream(
    context: ProviderDownloadContext,
    filename: string,
    mimeType: string,
  ): Promise<ProxyStreamResult> {
    let url: string;

    switch (context.provider) {
      case "r2": {
        url = await this.getR2PresignedUrl(context.identifier);
        break;
      }

      case "idrive": {
        url = await this.getIdrivePresignedUrl(context.identifier);
        break;
      }

      case "pixeldrain": {
        // Authenticated download endpoint — bypasses hotlink detection
        const apiKey = process.env.PIXELDRAIN_API_KEY || "";
        const credentials = Buffer.from(`:${apiKey}`).toString("base64");
        return this._streamFromUrl(
          `https://pixeldrain.com/api/file/${context.identifier}?download`,
          filename,
          mimeType,
          { Authorization: `Basic ${credentials}` },
        );
      }

      case "vikingfile": {
        // VikingFile's /f/{hash} is an HTML page – not a binary.
        // Their direct download endpoint pattern is: /download/{hash}
        // We verify the file exists first via the check-file API.
        const hash = context.identifier;
        const directDownloadUrl = `https://vikingfile.com/download/${hash}`;

        try {
          // Verify file existence via official API before streaming
          const checkRes = await axios.post(
            "https://vikingfile.com/api/check-file",
            new URLSearchParams({ hash }),
            {
              headers: { "Content-Type": "application/x-www-form-urlencoded" },
              timeout: 8000,
            },
          );
          if (!checkRes.data?.exist) {
            throw new Error(`VikingFile: file hash ${hash} does not exist`);
          }
        } catch (checkErr: any) {
          // Non-fatal: if check-file call itself fails (network etc.), still try to stream
          console.warn(
            "[DownloadService][VikingFile] check-file API failed:",
            checkErr.message,
          );
        }

        return this._streamFromUrl(directDownloadUrl, filename, mimeType);
      }

      default:
        throw new Error(`Unknown provider: ${context.provider}`);
    }

    return this._streamFromUrl(url, filename, mimeType);
  }

  /**
   * Opens an HTTP(S) stream from a URL with optional extra headers.
   */
  private static _streamFromUrl(
    url: string,
    filename: string,
    mimeType: string,
    extraHeaders: Record<string, string> = {},
  ): Promise<ProxyStreamResult> {
    return new Promise((resolve, reject) => {
      const lib = url.startsWith("https") ? https : http;

      const options = new URL(url);
      const reqOptions = {
        hostname: options.hostname,
        port: options.port,
        path: options.pathname + options.search,
        headers: {
          "User-Agent": "FileUploadProxy/1.0",
          ...extraHeaders,
        },
      };

      const req = lib.get(reqOptions, (res) => {
        // Follow redirects (max 3 hops)
        if (
          res.statusCode &&
          res.statusCode >= 300 &&
          res.statusCode < 400 &&
          res.headers.location
        ) {
          res.resume();
          this._streamFromUrl(
            res.headers.location,
            filename,
            mimeType,
            extraHeaders,
          )
            .then(resolve)
            .catch(reject);
          return;
        }

        if (!res.statusCode || res.statusCode >= 400) {
          res.resume();
          reject(
            new Error(
              `Upstream returned HTTP ${res.statusCode} for provider stream`,
            ),
          );
          return;
        }

        resolve({
          stream: res as unknown as Readable,
          contentType:
            res.headers["content-type"] ||
            mimeType ||
            "application/octet-stream",
          contentLength: res.headers["content-length"]
            ? Number(res.headers["content-length"])
            : undefined,
          filename,
        });
      });

      req.on("error", reject);
    });
  }
}
