import { prisma } from "../index";
import { extractDriveFileId } from "../utils/google-drive.utils";
import { Readable, PassThrough } from "stream";
import https from "https";
import { StorageProvider } from "./providers/storage-provider.interface";
import { R2Provider } from "./providers/r2.provider";
import { PixeldrainProvider } from "./providers/pixeldrain.provider";
import { VikingfileProvider } from "./providers/vikingfile.provider";
import { IdriveProvider } from "./providers/idrive.provider";

export class FileService {
  /**
   * Starts the async upload process.
   */
  static async uploadFromDrive(url: string) {
    const driveFileId = extractDriveFileId(url);
    if (!driveFileId) {
      throw new Error("Invalid Google Drive URL. Could not extract File ID.");
    }

    const activeProvidersString = process.env.ACTIVE_STORAGE_PROVIDERS || "R2";
    const activeProvidersEnum = activeProvidersString
      .split(",")
      .map((p) => p.trim().toUpperCase() as any); // Cast as Prisma enum later

    // Create pending database record
    const fileRecord = await prisma.file.create({
      data: {
        status: "PENDING",
        provider: activeProvidersEnum,
      },
    });

    // Start background process (fire and forget)
    this._processUpload(fileRecord.id, driveFileId).catch((error) => {
      console.error(`Background upload failed for ${fileRecord.id}:`, error);
    });

    return fileRecord;
  }

  /**
   * Internal background worker to handle streaming and DB updates.
   */
  private static async _processUpload(fileId: string, driveFileId: string) {
    const apiKey = process.env.GOOGLE_DRIVE_API_KEY;
    if (!apiKey) {
      await prisma.file.update({
        where: { id: fileId },
        data: { status: "FAILED" },
      });
      console.error("GOOGLE_DRIVE_API_KEY is not set");
      return;
    }

    try {
      // 1. Fetch Metadata (name, mimeType, size)
      const metaUrl = `https://www.googleapis.com/drive/v3/files/${driveFileId}?fields=id,name,mimeType,size&key=${apiKey}`;
      const metaResponse = await fetch(metaUrl);
      if (!metaResponse.ok) {
        throw new Error(
          `Failed to fetch metadata from Google Drive: ${metaResponse.statusText}`,
        );
      }
      const meta = await metaResponse.json();
      const fileSize = meta.size ? BigInt(meta.size) : null;

      await prisma.file.update({
        where: { id: fileId },
        data: {
          originalName: meta.name,
          mimeType: meta.mimeType,
          size: fileSize,
          status: "DOWNLOADING",
          progress: 0,
        },
      });

      // Initialize Providers
      const activeProvidersString =
        process.env.ACTIVE_STORAGE_PROVIDERS || "R2";
      const activeProviders = activeProvidersString
        .split(",")
        .map((p) => p.trim().toUpperCase());

      const providerInstances: { name: string; instance: StorageProvider }[] =
        [];
      if (activeProviders.includes("R2"))
        providerInstances.push({ name: "R2", instance: new R2Provider() });
      if (activeProviders.includes("PIXELDRAIN"))
        providerInstances.push({
          name: "PIXELDRAIN",
          instance: new PixeldrainProvider(),
        });
      if (activeProviders.includes("VIKINGFILE"))
        providerInstances.push({
          name: "VIKINGFILE",
          instance: new VikingfileProvider(),
        });
      if (activeProviders.includes("IDRIVE"))
        providerInstances.push({
          name: "IDRIVE",
          instance: new IdriveProvider(),
        });

      if (providerInstances.length === 0) {
        throw new Error(
          "No active storage providers configured. Set ACTIVE_STORAGE_PROVIDERS.",
        );
      }

      // 2. Fetch Media stream using native http to get Node.js stream directly
      const mediaUrl = `https://www.googleapis.com/drive/v3/files/${driveFileId}?alt=media&key=${apiKey}`;

      const sourceStream = await new Promise<Readable>((resolve, reject) => {
        https
          .get(mediaUrl, (res) => {
            if (res.statusCode !== 200) {
              reject(new Error(`Failed to stream file: ${res.statusCode}`));
              res.resume();
              return;
            }
            resolve(res);
          })
          .on("error", reject);
      });

      // Custom progress tracking proxy
      let loaded = 0;
      let lastProgressUpdate = 0;
      sourceStream.on("data", async (chunk) => {
        loaded += chunk.length;
        if (fileSize) {
          const percent = Math.floor((loaded / Number(fileSize)) * 100);
          if (percent > lastProgressUpdate || percent === 100) {
            lastProgressUpdate = percent;
            await prisma.file.update({
              where: { id: fileId },
              data: { progress: percent },
            });
          }
        }
      });

      // 3. Orchestrate concurrent uploads
      // Create a PassThrough for each provider so they can consume the same source HTTP stream
      const uploadPromises = providerInstances.map(async (providerObj) => {
        const pt = new PassThrough();
        sourceStream.pipe(pt);

        const identifier = await providerObj.instance.uploadStream(pt, {
          name: meta.name || "file.bin",
          mimeType: meta.mimeType || "application/octet-stream",
          size: Number(fileSize) || undefined,
        });

        return { name: providerObj.name, identifier };
      });

      const results = await Promise.all(uploadPromises);

      // 4. Mark Completed and save IDs
      const updateData: any = {
        status: "COMPLETED",
        progress: 100,
        provider: activeProviders,
      };

      for (const result of results) {
        if (!result.identifier) continue;

        if (result.name === "R2") {
          updateData.r2Key = result.identifier.identifier;
          updateData.r2Url = result.identifier.url;
        }
        if (result.name === "PIXELDRAIN") {
          updateData.pixeldrainId = result.identifier.identifier;
          updateData.pixeldrainUrl = result.identifier.url;
        }
        if (result.name === "VIKINGFILE") {
          updateData.vikingfileId = result.identifier.identifier;
          updateData.vikingfileUrl = result.identifier.url;
        }
        if (result.name === "IDRIVE") {
          updateData.idriveKey = result.identifier.identifier;
          updateData.idriveUrl = result.identifier.url;
        }
      }

      await prisma.file.update({
        where: { id: fileId },
        data: updateData,
      });

      console.log(
        `Successfully uploaded file DB ID: ${fileId} via active providers.`,
      );
    } catch (error) {
      console.error(`Error processing file ${fileId}:`, error);
      await prisma.file.update({
        where: { id: fileId },
        data: { status: "FAILED" },
      });
    }
  }

  static async getAllFiles() {
    return prisma.file.findMany({
      orderBy: { createdAt: "desc" },
    });
  }

  static async getFile(id: string) {
    return prisma.file.findUnique({
      where: { id },
    });
  }

  static async deleteFile(id: string) {
    console.log(`[FileService Delete] Attempting to delete file ID: ${id}`);
    const file = await prisma.file.findUnique({ where: { id } });

    if (!file) {
      console.warn(`[FileService Delete] File ID ${id} not found in database.`);
      throw new Error("File not found");
    }

    const deletePromises = [];

    if (file.r2Key) {
      const p = new R2Provider()
        .deleteFile(file.r2Key)
        .catch((err) => console.error("R2 delete Error:", err));
      deletePromises.push(p);
    }

    if (file.pixeldrainId) {
      const p = new PixeldrainProvider()
        .deleteFile(file.pixeldrainId)
        .catch((err) => console.error("Pixeldrain delete Error:", err));
      deletePromises.push(p);
    }

    if (file.vikingfileId) {
      const p = new VikingfileProvider()
        .deleteFile(file.vikingfileId)
        .catch((err) => console.error("Vikingfile delete Error:", err));
      deletePromises.push(p);
    }

    if (file.idriveKey) {
      const p = new IdriveProvider()
        .deleteFile(file.idriveKey)
        .catch((err) => console.error("IDrive delete Error:", err));
      deletePromises.push(p);
    }

    await Promise.all(deletePromises);

    await prisma.file.delete({
      where: { id },
    });
    console.log(`[FileService Delete] Database record deleted successfully.`);

    return true;
  }
}
