import { StorageProvider } from "./storage-provider.interface";
import { Readable } from "stream";
import { Upload } from "@aws-sdk/lib-storage";
import { DeleteObjectCommand } from "@aws-sdk/client-s3";
import { idriveClient, IDRIVE_BUCKET_NAME } from "../../config/idrive.config";
import { v4 as uuidv4 } from "uuid";

export class IdriveProvider implements StorageProvider {
  /**
   * Uploads stream to IDrive e2 (S3 compatible)
   */
  async uploadStream(
    stream: Readable,
    meta: { name: string; mimeType: string; size?: number },
  ): Promise<{ identifier: string; url: string } | null> {
    try {
      const idriveKey = `${uuidv4()}-${meta.name || "file"}`;
      const upload = new Upload({
        client: idriveClient,
        params: {
          Bucket: IDRIVE_BUCKET_NAME,
          Key: idriveKey,
          Body: stream,
          ContentType: meta.mimeType || "application/octet-stream",
        },
      });

      await upload.done();

      // IDrive endpoints usually start with https://, so we securely build the URL
      const endpoint =
        process.env.IDRIVE_ENDPOINT || "https://s3.eu-west-3.idrivee2.com";
      const cleanEndpoint = endpoint.replace(/^https?:\/\//, "");
      const publicUrl = `https://${cleanEndpoint}/${IDRIVE_BUCKET_NAME}/${idriveKey}`;

      return { identifier: idriveKey, url: publicUrl };
    } catch (error) {
      console.error("IDrive upload error:", error);
      throw new Error("Failed to upload to IDrive");
    }
  }

  async deleteFile(identifier: string): Promise<boolean> {
    try {
      await idriveClient.send(
        new DeleteObjectCommand({
          Bucket: IDRIVE_BUCKET_NAME,
          Key: identifier,
        }),
      );
      return true;
    } catch (err) {
      console.error("Failed to delete from IDrive:", err);
      return false;
    }
  }
}
