import { StorageProvider } from "./storage-provider.interface";
import { Readable } from "stream";
import { Upload } from "@aws-sdk/lib-storage";
import { DeleteObjectCommand } from "@aws-sdk/client-s3";
import { s3Client, R2_BUCKET_NAME } from "../../config/r2.config";
import { v4 as uuidv4 } from "uuid";

export class R2Provider implements StorageProvider {
  /**
   * We can accept a progress callback here or handle it externally.
   * For simplicity, returning the identifier (r2Key).
   */
  async uploadStream(
    stream: Readable,
    meta: { name: string; mimeType: string; size?: number },
  ): Promise<{ identifier: string; url: string } | null> {
    try {
      const r2Key = `${uuidv4()}-${meta.name || "file"}`;
      const upload = new Upload({
        client: s3Client,
        params: {
          Bucket: R2_BUCKET_NAME,
          Key: r2Key,
          Body: stream,
          ContentType: meta.mimeType || "application/octet-stream",
        },
      });

      // We omit individual progress here to keep the interface simple,
      // but the stream consumer can track progress if we wrap the stream
      // before passing it to uploadStream().

      await upload.done();
      const publicUrl = process.env.R2_PUBLIC_URL
        ? `${process.env.R2_PUBLIC_URL}/${r2Key}`
        : `https://${R2_BUCKET_NAME}.r2.cloudflarestorage.com/${r2Key}`;
      return { identifier: r2Key, url: publicUrl };
    } catch (error) {
      console.error("R2 upload error:", error);
      throw new Error("Failed to upload to R2");
    }
  }

  async deleteFile(identifier: string): Promise<boolean> {
    try {
      await s3Client.send(
        new DeleteObjectCommand({
          Bucket: R2_BUCKET_NAME,
          Key: identifier,
        }),
      );
      return true;
    } catch (err) {
      console.error("Failed to delete from R2:", err);
      return false;
    }
  }
}
