import { StorageProvider } from "./storage-provider.interface";
import { Readable } from "stream";
import FormData from "form-data";
import axios from "axios";
import * as fs from "fs";
import * as os from "os";
import * as path from "path";
import { pipeline } from "stream/promises";

export class VikingfileProvider implements StorageProvider {
  private readonly userHash: string;

  constructor() {
    this.userHash =
      process.env.VIKINGFILE_USER_HASH || process.env.VIKINGFILE_API_KEY || "";
  }

  /**
   * Uploads a stream to VikingFile.
   *
   * We first buffer the stream to a temp file, then upload from the file.
   * This is necessary because VikingFile's multipart upload via axios does not
   * work reliably with a piped PassThrough stream when multiple providers are
   * uploading concurrently from the same source — the stream may be drained
   * before VikingFile finishes reading it.
   */
  async uploadStream(
    stream: Readable,
    meta: { name: string; mimeType: string; size?: number },
  ): Promise<{ identifier: string; url: string } | null> {
    // Write stream to a temp file so we have a stable, seekable input
    const tmpPath = path.join(
      os.tmpdir(),
      `vf-upload-${Date.now()}-${meta.name}`,
    );

    try {
      const writeStream = fs.createWriteStream(tmpPath);
      await pipeline(stream, writeStream);

      let serverUrl = "https://upload.vikingfile.com";
      try {
        const res = await axios.get("https://vikingfile.com/api/get-server", {
          timeout: 5000,
        });
        if (res.data?.server) {
          serverUrl = res.data.server;
        }
      } catch {
        console.warn(
          "[VikingFile] Could not fetch upload server, using default",
        );
      }

      const fileStream = fs.createReadStream(tmpPath);
      const fileSize = fs.statSync(tmpPath).size;

      const form = new FormData();
      if (this.userHash) {
        form.append("user", this.userHash);
      }
      form.append("file", fileStream, {
        filename: meta.name || "file.bin",
        contentType: meta.mimeType || "application/octet-stream",
        knownLength: fileSize,
      });

      const response = await axios.post(serverUrl, form, {
        headers: {
          ...form.getHeaders(),
        },
        maxBodyLength: Infinity,
        maxContentLength: Infinity,
        timeout: 0, // No timeout for large file uploads
      });

      if (response.data && (response.data.hash || response.data.url)) {
        const hash = response.data.hash;
        const publicUrl =
          response.data.url || `https://vikingfile.com/f/${hash}`;
        return { identifier: hash, url: publicUrl };
      }

      console.error("[VikingFile] Unexpected upload response:", response.data);
      return null;
    } catch (error: any) {
      console.error(
        "[VikingFile] Upload error:",
        error.response?.data || error.message,
      );
      throw new Error("Failed to upload to VikingFile");
    } finally {
      // Always clean up temp file
      fs.unlink(tmpPath, (err) => {
        if (err)
          console.warn(
            "[VikingFile] Failed to clean up temp file:",
            err.message,
          );
      });
    }
  }

  async deleteFile(identifier: string): Promise<boolean> {
    if (!this.userHash) {
      console.warn("[VikingFile] Delete skipped: No user hash provided.");
      return false;
    }

    try {
      const form = new FormData();
      form.append("hash", identifier);
      form.append("user", this.userHash);
      const response = await axios.post(
        "https://vikingfile.com/api/delete-file",
        form,
        { headers: { ...form.getHeaders() } },
      );

      if (response.data?.error === "success") {
        return true;
      }
      console.error("[VikingFile] Delete failed:", response.data);
      return false;
    } catch (error: any) {
      console.error(
        "[VikingFile] Delete error:",
        error.response?.data || error.message,
      );
      return false;
    }
  }
}
